<?php
/**
 * Elementor Widget
 * @package Foresty
 * @since 1.0.0
 */

namespace Elementor;
class Foresty_Section_Title_One_Widget extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'foresty-theme-heading-title-one-widget';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Heading Title: 01', 'foresty-core');
    }

    public function get_keywords()
    {
        return ['Section', 'Heading', 'Title', "Gramentheme", 'Foresty'];
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-heading';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['foresty_widgets'];
    }

    /**
     * Register Elementor widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {

        $this->start_controls_section(
            'settings_section',
            [
                'label' => esc_html__('General Settings', 'foresty-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'subtitle_plane_animation',
            [
                'label' => esc_html__('Subtitle Plane Animation', 'foresty-core'),
                'type' => Controls_Manager::SELECT,
                'default' => '',
                'options' => [
                    'shape' => esc_html__('With Plane', 'foresty-core'),
                    '' => esc_html__('Without Plane', 'foresty-core'),
                ],
            ]
        );
        $this->add_control(
            'plane_icon_left',
            [
                'label' => esc_html__('Animation Icon Left', 'foresty-core'),
                'type' => Controls_Manager::ICONS,
                'description' => esc_html__('select Icon.', 'foresty-core'),
                'default' => [
                    'value' => 'flaticon-black-plane',
                    'library' => 'solid',
                ],
                'condition' => ['subtitle_plane_animation' => 'shape']
            ]
        );
        $this->add_control(
            'subtitle',
            [
                'label' => esc_html__('Sub Title', 'foresty-core'),
                'type' => Controls_Manager::TEXTAREA,
                'default' => esc_html__('About Foresty', 'foresty-core'),
                'description' => esc_html__('enter title. use {c} color text {/c} for color text', 'foresty-core'),
            ]
        );
        $this->add_control(
            'title',
            [
                'label' => esc_html__('Title', 'foresty-core'),
                'type' => Controls_Manager::TEXTAREA,
                'default' => esc_html__('What We Do', 'foresty-core'),
            ]
        );
        $this->add_control(
            'description_status',
            [
                'label' => esc_html__('Description Show/Hide', 'foresty-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('show/hide description', 'foresty-core'),
            ]
        );
        $this->add_control(
            'description',
            [
                'label' => esc_html__('Description', 'foresty-core'),
                'type' => Controls_Manager::TEXTAREA,
                'description' => esc_html__('enter  description.', 'foresty-core'),
                'default' => esc_html__('Top Packages', 'foresty-core'),
                'condition' => ['description_status' => 'yes']
            ]
        );
        $this->add_control(
            'btn_status',
            [
                'label' => esc_html__('btn Show/Hide', 'foresty-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('show/hide description', 'foresty-core'),
            ]
        );
        $this->add_control(
            'btn_class',
            [
                'label' => esc_html__('btn class', 'foresty-core'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('btn-one-light', 'foresty-core'),
                'condition' => ['btn_status' => 'yes']
            ]
        );
        $this->add_control(
            'btn_text',
            [
                'label' => esc_html__('btn', 'foresty-core'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Join With Us', 'foresty-core'),
                'condition' => ['btn_status' => 'yes']
            ]
        );
        $this->add_control(
            'btn_url',
            [
                'label' => esc_html__('btn url', 'foresty-core'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('#', 'foresty-core'),
                'condition' => ['btn_status' => 'yes']
            ]
        );
        $this->add_responsive_control(
            'text_align',
            [
                'label' => esc_html__('Alignment', 'foresty-core'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__('Left', 'foresty-core'),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__('Center', 'foresty-core'),
                        'icon' => 'fa fa-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'foresty-core'),
                        'icon' => 'fa fa-align-right',
                    ],
                ],
                'default' => 'center',
                'toggle' => true,
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'styling_section',
            [
                'label' => esc_html__('Styling Settings', 'foresty-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'shape_top_space',
            [
                'label' => esc_html__('Shape Top Space', 'foresty-core'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 5,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}}.theme-heading-title .title.shape' => 'padding-top: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'title_bottom_space',
            [
                'label' => esc_html__('Title Bottom Space', 'foresty-core'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 5,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .theme-heading-title .title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control('subtitle_color', [
            'label' => esc_html__('Sub Title Color', 'foresty-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .theme-heading-title .subtitle" => "color: {{VALUE}}"
            ]
        ]);
        $this->add_control('subtitle_extra_color', [
            'label' => esc_html__('Sub Title Extra Color', 'foresty-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .theme-heading-title .subtitle span" => "color: {{VALUE}}"
            ]
        ]);
        $this->add_control('description_color', [
            'label' => esc_html__('Description Color', 'foresty-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .theme-heading-title p" => "color: {{VALUE}}"
            ]
        ]);

        $this->add_control('title_color', [
            'label' => esc_html__('Title Color', 'foresty-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .theme-heading-title .title" => "color: {{VALUE}}"
            ]
        ]);
        $this->add_control('title_extra_color', [
            'label' => esc_html__('Title Extra Color', 'foresty-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .theme-heading-title .title span" => "color: {{VALUE}}"
            ]
        ]);

        $this->end_controls_section();
        $this->start_controls_section(
            'styling_typogrpahy_section',
            [
                'label' => esc_html__('Typography Settings', 'foresty-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_group_control(Group_Control_Typography::get_type(), [
            'name' => 'sub_title_typography',
            'label' => esc_html__('Title Typography', 'foresty-core'),
            'selector' => "{{WRAPPER}} .theme-heading-title .subtitle"
        ]);
        $this->add_group_control(Group_Control_Typography::get_type(), [
            'name' => 'title_typography',
            'label' => esc_html__('Title Typography', 'foresty-core'),
            'selector' => "{{WRAPPER}} .theme-heading-title .title"
        ]);
        $this->add_group_control(Group_Control_Typography::get_type(), [
            'name' => 'title_extra_typography',
            'label' => esc_html__('Title Extra Typography', 'foresty-core'),
            'selector' => "{{WRAPPER}} .theme-heading-title .title span"
        ]);
        $this->add_group_control(Group_Control_Typography::get_type(), [
            'name' => 'description_typography',
            'label' => esc_html__('Description Typography', 'foresty-core'),
            'selector' => "{{WRAPPER}} .theme-heading-title p"
        ]);
        $this->end_controls_section();
    }

    /**
     * Render Elementor widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
        ?>
        <div class="theme-heading-title" style="text-align:<?php echo $settings['text_align']; ?>">
            <?php if (!empty($settings['subtitle'])) { ?>
                <div class="subtitle <?php echo $settings['subtitle_plane_animation'] ?>">
                    <?php if (!empty($settings['plane_icon_left'])): ?>
                        <div class="icon-left d-inline-block">
                            <?php
                            Icons_Manager::render_icon($settings['plane_icon_left'], ['aria-hidden' => 'true']);
                            ?>
                        </div>
                    <?php endif; ?>
                    <?php
                    $subtitle = str_replace(['{c}', '{/c}'], ['<span>', '</span>'], $settings['subtitle']);
                    print wp_kses($subtitle, foresty_core()->kses_allowed_html('all'));
                    ?>
                </div>
            <?php } ?>
            <h3 class="title">
                <?php
                $title = str_replace(['{c}', '{/c}'], ['<span>', '</span>'], $settings['title']);
                print wp_kses($title, foresty_core()->kses_allowed_html('all'));
                ?>
            </h3>
            <?php
            if (!empty($settings['description_status'])) {
                printf('<p>%1$s</p>', esc_html($settings['description']));
            }
            if (!empty($settings['btn_status'])) { ?>
                <div class="btn-wrap mt-xl-5 mt-4">
                    <a href="<?php echo $settings['btn_url']; ?>" class="<?php echo $settings['btn_class']; ?>"><span><?php echo $settings['btn_text']; ?></span> <i class="fa-solid fa-angles-right"></i></a>
                </div>
            <?php }
            ?>
        </div>
        <?php
    }
}

Plugin::instance()->widgets_manager->register_widget_type(new Foresty_Section_Title_One_Widget());